import numpy as np
import matplotlib
#matplotlib.rcParams['text.usetex'] = True
#matplotlib.rcParams['text.latex.preamble'] = r'\usepackage{libertine}'
#matplotlib.rcParams['font.family'] = "libertinus"
#params = {'text.latex.preamble': [r'\usepackage{libertine}', 
#    r'\usepackage{upgreek}']}   
matplotlib.rcParams['axes.linewidth'] = 0.5

import matplotlib.pyplot as plt
#plt.rcParams.update(params)

from mpl_toolkits.axes_grid1.inset_locator import inset_axes
import pandas as pd
import qcodes as qc
#Made by Damian Bouwmeester
#This file defines a starting point for plots to be put in a PhD thesis or journal.
#Certain standards are put in place that produce nice/decent looking images without a lot of work.
#Standard size is 89mm(single column) or 183mm(double column) for most journals dpi of 300-600 at highest quality recommended




def toArray(A):
    B=np.zeros((len(A),len(A[0])))
    for i in range(len(A)):
        for j in range(len(A[0])):
            B[i,j]=A[i][j]
    return B

def loaddata(numid):
    dataset=qc.load_by_id(numid)
    df=dataset.to_pandas_dataframe_dict()
    df=pd.concat(df,axis=1)
    DataNP=np.array(df.to_records().view(type=np.ndarray))
    DataArray=toArray(DataNP)
    return DataArray

#18pt
#8pt for legends and labels 


class DPlot():
    
    def __init__(self,size=3,dpi=300):
        self.cmtoinch=1/2.54
        
        if size==1:
            # Big figure, 1x in page., 1 in a page
            figw=12.3
            figh=12.3
        
        if size==2:
            #medium figures 2x in page, 4 in a page
            figw=12.3/2
            figh=12.3/2
        
        if size==3:
            #Small figures, 3x in page width, 12 in a page
            figw=12.3/3
            figh=12.3/3
         
        
        if size==4:
            #Rectangular figure 2x in page width, 8 in a page
            figw=12.3/2
            figh=12.3/3
            
        if size==5:
            #Small figures. 4x in page width, 16 or 20 in a page. (?)
            figw=12.3/4
            figh=12.3/4
        
        
        self.fig=plt.figure(figsize=(figw*self.cmtoinch,figh*self.cmtoinch),dpi=dpi)
        self.ax=self.fig.gca()
        self.ax.tick_params(direction='in', length=3, width=0.5, colors='k',
               grid_color='k', grid_alpha=0.5)
    
        self.scales=['f','p','n','$\mu$','m','','k','M','G','T']
    
        self.scale=1
        self.scale_label=''
        self.xscale=1
        self.xscale_label=''
        self.yscale=1
        self.yscale_label=''
        self.zscale=1
        self.zscale_label=''
        
        self.scaled=0
        
        
        self.inset_scale=1
        self.inset_scale_label=''
        self.inset_xscale=1
        self.inset_xscale_label=''
        self.inset_yscale=1
        self.inset_yscale_label=''
        self.inset_zscale=1
        self.inset_zscale_label=''
        
        self.inset_scaled=0
        
        self.fontsize=7.0 #Font size for axes and
        self.labelsize=5.0 #Size of labels(ticks, etc)
        self.fontsizeout=8.5 #Font size for text outside plots
        self.fontsizein=6.0 #Font size for text in plots
        
        self.pad=0.2       #padding around plots in tight_layout
    
    

    def plot2D(self,x,y,xlabel,ylabel,xunit,yunit,marker='',mfc='k',mec='k',mew=0,linestyle='-',color=None,markersize=5,linewidth=1,fontsize=8.5,labelsize=6,label=None,rescale=True):
        
        fontsize=self.fontsize
        labelsize=self.labelsize
        
        # If first plot
        if self.scaled==0 and rescale:
            self.set_varscalex(x)
            self.set_varscaley(y)
    
            self.ax.set_xlabel(xlabel=xlabel+' ('+self.xscale_label+xunit+')',fontsize=fontsize)
            self.ax.set_ylabel(ylabel=ylabel+' ('+self.yscale_label+yunit+')',fontsize=fontsize)
            self.ax.tick_params(labelsize=labelsize,axis='x')
            self.ax.tick_params(labelsize=labelsize,axis='y')
            self.scaled=1 #Lock the scaling
        
        if rescale:
            xplot=self.varscalex(x)
            yplot=self.varscaley(y)
        else:
            xplot=x
            yplot=y
            self.ax.set_xlabel(xlabel=xlabel+' ('+xunit+')',fontsize=fontsize)
            self.ax.set_ylabel(ylabel=ylabel+' ('+yunit+')',fontsize=fontsize)
            self.ax.tick_params(labelsize=labelsize,axis='x')
            self.ax.tick_params(labelsize=labelsize,axis='y')
            
        if (mfc==None and mec==None):
            self.ax.plot(xplot,yplot,marker=marker,mew=mew,linestyle=linestyle,markersize=markersize,linewidth=linewidth,c=color, label=label)
        else:
            self.ax.plot(xplot,yplot,marker=marker,mfc=mfc,mec=mec,mew=mew,linestyle=linestyle, markersize=markersize,linewidth=linewidth,c=color, label=label)
        
        #plt.ticklabel_format(axis="x", style="plain", scilimits=(0,0))
        #plt.ticklabel_format(axis="y", style="plain", scilimits=(0,0))
        self.ax.locator_params(axis='x', nbins=5)
        
        self.fig.tight_layout(pad=self.pad)
       
    
    def plot2Dhistlogx(self,x,bins,xlabel,ylabel,xunit,fontsize=8.5,labelsize=6, edgecolor='black', linewidth=1.2):
        
        fontsize=self.fontsize
        labelsize=self.labelsize
        
        self.set_varscalex(x)
        xplot=self.varscalex(x)       
        
        self.ax.set_xlabel(xlabel=xlabel+' ('+self.xscale_label+xunit+')',fontsize=fontsize)
        self.ax.set_ylabel(ylabel=ylabel,fontsize=fontsize)
        self.ax.tick_params(labelsize=labelsize,axis='x')
        self.ax.tick_params(labelsize=labelsize,axis='y')
        
        logbins = np.geomspace(xplot.min(), xplot.max(), bins)
        
        self.ax.hist(xplot,bins=logbins, edgecolor='black', linewidth=1.2)
        
        self.ax.set_xscale('log')
        
        #plt.ticklabel_format(axis="x", style="plain", scilimits=(0,0))
        #plt.ticklabel_format(axis="y", style="plain", scilimits=(0,0))
        #plt.locator_params(axis='x', nbins=10)
        
        self.fig.tight_layout(pad=self.pad)
        
    def CreateAdjacent(self,nx,ny,sharex=False,sharey=True,hspace=0,vspace=0,figw=12.3,figh=12.3,dpi=300,left=None, bottom=None, right=None, top=None):
        
        self.vspace=vspace
        self.hspace=hspace
        
        self.fig, self.axs = plt.subplots(ny,nx, sharex=sharex,sharey=sharey,figsize=(figw/ny*self.cmtoinch,figh/nx*self.cmtoinch),dpi=dpi)
        # Remove horizontal space between axes
        self.fig.subplots_adjust(hspace=vspace,wspace=hspace,left=left, bottom=bottom, right=right, top=top)
    
    
    def plot2DAdjacent(self,i,x,y,xlabel,ylabel,xunit,yunit,marker='',mfc='k',mec='k',mew=0,linestyle='-',color=None,markersize=5,linewidth=1,fontsize=8.5,labelsize=6,label=None,rescale=True):
        
        fontsize=self.fontsize
        labelsize=self.labelsize
        
        # If first plot
        if self.scaled==0 and rescale:
            self.set_varscalex(x)
            self.set_varscaley(y)
    
            self.scaled=1 #Lock the scaling
        
        if xlabel!=None:
            self.axs[i].set_xlabel(xlabel=xlabel+' ('+self.xscale_label+xunit+')',fontsize=fontsize)
        if ylabel!=None:
            self.axs[i].set_ylabel(ylabel=ylabel+' ('+self.yscale_label+yunit+')',fontsize=fontsize)
        self.axs[i].tick_params(labelsize=labelsize,axis='x')
        self.axs[i].tick_params(labelsize=labelsize,axis='y')
        if rescale:
            xplot=self.varscalex(x)
            yplot=self.varscaley(y)
        else:
            xplot=x
            yplot=y
        
        if (mfc==None and mec==None):
            self.axs[i].plot(xplot,yplot,marker=marker,mew=mew,linestyle=linestyle,markersize=markersize,linewidth=linewidth,c=color, label=label)
        else:
            self.axs[i].plot(xplot,yplot,marker=marker,mfc=mfc,mec=mec,mew=mew,linestyle=linestyle, markersize=markersize,linewidth=linewidth,c=color, label=label)
        
        #plt.ticklabel_format(axis="x", style="plain", scilimits=(0,0))
        #plt.ticklabel_format(axis="y", style="plain", scilimits=(0,0))
        self.axs[i].locator_params(axis='x', nbins=5)
        
        self.fig.subplots_adjust(top=0.9,bottom=0.2,hspace=self.vspace,wspace=self.hspace)
    
    def plot2DAdjacent2(self,i,j,x,y,xlabel,ylabel,xunit,yunit,marker='',mfc='k',mec='k',mew=0,linestyle='-',color=None,markersize=5,linewidth=1,fontsize=8.5,labelsize=6,label=None,rescale=True):
        
        fontsize=self.fontsize
        labelsize=self.labelsize
        
        # If first plot
        if self.scaled==0 and rescale:
            self.set_varscalex(x)
            self.set_varscaley(y)
    
            self.scaled=1 #Lock the scaling
        
        self.axs[i,j].set_xlabel(xlabel=xlabel+' ('+self.xscale_label+xunit+')',fontsize=fontsize)
        self.axs[i,j].set_ylabel(ylabel=ylabel+' ('+self.yscale_label+yunit+')',fontsize=fontsize)
        self.axs[i,j].tick_params(labelsize=labelsize,axis='x')
        self.axs[i,j].tick_params(labelsize=labelsize,axis='y')
        
        if rescale:
            xplot=self.varscalex(x)
            yplot=self.varscaley(y)
        else:
            xplot=x
            yplot=y
        
        if (mfc==None and mec==None):
            self.axs[i,j].plot(xplot,yplot,marker=marker,mew=mew,linestyle=linestyle,markersize=markersize,linewidth=linewidth,c=color, label=label)
        else:
            self.axs[i,j].plot(xplot,yplot,marker=marker,mfc=mfc,mec=mec,mew=mew,linestyle=linestyle, markersize=markersize,linewidth=linewidth,c=color, label=label)
        
        #plt.ticklabel_format(axis="x", style="plain", scilimits=(0,0))
        #plt.ticklabel_format(axis="y", style="plain", scilimits=(0,0))
        self.axs[i,j].locator_params(axis='x', nbins=5)
        
        self.fig.subplots_adjust(hspace=self.vspace,wspace=self.hspace)
    
    
    def plot3DAdjacent(self,i,x,y,z,xlabel,ylabel,clabel,xunit,yunit,cunit,fontsize=8.5,labelsize=6,vmin=None,vmax=None,cmap='inferno',shading='auto',cbar=True,rescale_ax=True, rescalez=True, labelx=True,labely=True):
        
        fontsize=self.fontsize
        labelsize=self.labelsize
        
        if rescale_ax==True:
            self.set_varscalex(x)
            self.set_varscaley(y)
            xplot=self.varscalex(x)       
            yplot=self.varscaley(y)
        else:
            xplot=x
            yplot=y
        
        if rescalez==True:
            self.set_varscalez(z)
            zplot=self.varscalez(z)
        else:
            zplot=z
        
        if labelx==True:
            self.axs[i].set_xlabel(xlabel=xlabel+' ('+self.xscale_label+xunit+')',fontsize=fontsize)
        if labely==True:
            self.axs[i].set_ylabel(ylabel=ylabel+' ('+self.yscale_label+yunit+')',fontsize=fontsize)
        self.axs[i].tick_params(labelsize=labelsize,axis='x')
        self.axs[i].tick_params(labelsize=labelsize,axis='y')
        self.axs[i].tick_params(direction='out', length=1, width=0.5, colors='k',
               grid_color='k', grid_alpha=0.5)
        
        
        X,Y,Z=self.convert3D_to_matrix(xplot,yplot,zplot)
        self.p=self.axs[i].pcolormesh(X,Y,Z,cmap=cmap,vmin=vmin,vmax=vmax,shading=shading)
        
    def createcolorbarAdjacent(self,clabel,cunit,rescalez=True,x=0.67,y=0.15,w=0.05,h=0.7):
        fontsize=self.fontsize
        labelsize=self.labelsize
         
        cbar_ax = self.fig.add_axes([x, y, w, h])
        self.cb=self.fig.colorbar(self.p,cax=cbar_ax)
        self.cb.ax.get_yaxis().labelpad =fontsize*1.5
        if rescalez==True:
            self.cb.ax.set_ylabel(r''+clabel+' ('+self.zscale_label+cunit+')',fontsize=fontsize, rotation=270)
        else:
            self.cb.ax.set_ylabel(r''+clabel+' ('+cunit+')',fontsize=fontsize, rotation=270)
        self.cb.ax.tick_params(labelsize=labelsize)
        self.cb.ax.tick_params(direction='out', length=1, width=0.5, colors='k',
               grid_color='k', grid_alpha=0.5)
            #self.cb.formatter.set_powerlimits((0, 0))
        
        #plt.ticklabel_format(axis="x", style="plain", scilimits=(0,0))
        #plt.ticklabel_format(axis="y", style="plain", scilimits=(0,0))
        
        #self.fig.tight_layout()
    
    def createtwin(self,cleft='black',cright='red'):
        self.axtwin = self.ax.twinx()

        #Set colors
        self.axtwin.spines["left"].set_edgecolor(cleft)
        self.ax.yaxis.label.set_color(cleft)
        self.ax.tick_params(axis='y', colors=cleft)
        self.axtwin.spines["right"].set_edgecolor(cright)
        self.axtwin.yaxis.label.set_color(cright)
        self.axtwin.tick_params(axis='y', colors=cright)
        self.scaledtwin=0

    def plot2Dtwin(self,x,y,xlabel,ylabel,xunit,yunit,marker='',mfc='k',mec='k',mew=0,linestyle='-',color=None,markersize=5,linewidth=1,fontsize=8.5,labelsize=6,label=None,rescale=True):
        
        fontsize=self.fontsize
        labelsize=self.labelsize
        
        # If first plot
        if self.scaledtwin==0 and rescale:
            self.set_varscalex(x)
            self.set_varscaley(y)
    
            self.axtwin.set_xlabel(xlabel=xlabel+' ('+self.xscale_label+xunit+')',fontsize=fontsize)
            self.axtwin.set_ylabel(ylabel=ylabel+' ('+self.yscale_label+yunit+')',fontsize=fontsize)
            self.axtwin.tick_params(labelsize=labelsize,axis='x')
            self.axtwin.tick_params(labelsize=labelsize,axis='y')
            self.scaledtwin=1 #Lock the scaling
        
        if rescale:
            xplot=self.varscalex(x)
            yplot=self.varscaley(y)
        else:
            xplot=x
            yplot=y
            self.axtwin.set_xlabel(xlabel=xlabel+' ('+xunit+')',fontsize=fontsize)
            self.axtwin.set_ylabel(ylabel=ylabel+' ('+yunit+')',fontsize=fontsize)
            self.axtwin.tick_params(labelsize=labelsize,axis='x')
            self.axtwin.tick_params(labelsize=labelsize,axis='y')
            
        if (mfc==None and mec==None):
            self.axtwin.plot(xplot,yplot,marker=marker,mew=mew,linestyle=linestyle,markersize=markersize,linewidth=linewidth,c=color, label=label)
        else:
            self.axtwin.plot(xplot,yplot,marker=marker,mfc=mfc,mec=mec,mew=mew,linestyle=linestyle, markersize=markersize,linewidth=linewidth,c=color, label=label)
        
        #plt.ticklabel_format(axis="x", style="plain", scilimits=(0,0))
        #plt.ticklabel_format(axis="y", style="plain", scilimits=(0,0))
        self.axtwin.locator_params(axis='x', nbins=5)
        
        self.fig.tight_layout(pad=self.pad)
    
        
    def plot3D(self,x,y,z,xlabel,ylabel,clabel,xunit,yunit,cunit,fontsize=8.5,labelsize=6,vmin=None,vmax=None,cmap='inferno',shading='auto',cbar=True,rescale_ax=True, rescalez=True):
        
        fontsize=self.fontsize
        labelsize=self.labelsize
        
        if rescale_ax==True:
            self.set_varscalex(x)
            self.set_varscaley(y)
            xplot=self.varscalex(x)       
            yplot=self.varscaley(y)
        else:
            xplot=x
            yplot=y
        
        if rescalez==True:
            self.set_varscalez(z)
            zplot=self.varscalez(z)
        else:
            zplot=z
        
        self.ax.set_xlabel(xlabel=xlabel+' ('+self.xscale_label+xunit+')',fontsize=fontsize)
        self.ax.set_ylabel(ylabel=ylabel+' ('+self.yscale_label+yunit+')',fontsize=fontsize)
        self.ax.tick_params(labelsize=labelsize,axis='x')
        self.ax.tick_params(labelsize=labelsize,axis='y')
        self.ax.tick_params(labelsize=labelsize,axis='x')
        self.ax.tick_params(labelsize=labelsize,axis='y')
        self.ax.tick_params(direction='out', length=1, width=0.5, colors='k',
               grid_color='k', grid_alpha=0.5)
        
        
        X,Y,Z=self.convert3D_to_matrix(xplot,yplot,zplot)
        p=self.ax.pcolormesh(X,Y,Z,cmap=cmap,vmin=vmin,vmax=vmax,shading=shading)
        
        if cbar==True:
            self.cb=plt.colorbar(p)
            self.cb.ax.get_yaxis().labelpad = fontsize*1.5
            if rescalez==True:
                self.cb.ax.set_ylabel(r''+clabel+' ('+self.zscale_label+cunit+')',fontsize=fontsize, rotation=270)
            else:
                self.cb.ax.set_ylabel(r''+clabel+' ('+cunit+')',fontsize=fontsize, rotation=270)
            self.cb.ax.tick_params(labelsize=labelsize)
            self.cb.ax.tick_params(direction='out', length=1, width=0.5, colors='k',
               grid_color='k', grid_alpha=0.5)
            #self.cb.formatter.set_powerlimits((0, 0))
            print('Okay')
        
        #plt.ticklabel_format(axis="x", style="plain", scilimits=(0,0))
        #plt.ticklabel_format(axis="y", style="plain", scilimits=(0,0))
        
        self.fig.tight_layout(pad=self.pad)
    
    
    def set_varscale(self,x):
        if max(abs(x))<4e-13:
            self.scale_label=self.scales[0]
            self.scale=1e-15
        elif max(abs(x))<4e-10:
            self.scale_label=self.scales[1]
            self.scale=1e-12
        elif max(abs(x))<4e-7:
            self.scale_label=self.scales[2]
            self.scale=1e-9
        elif max(abs(x))<4e-4:
            self.scale_label=self.scales[3]
            self.scale=1e-6
        elif max(abs(x))<4e-1:
            self.scale_label=self.scales[4]
            self.scale=1e-3
        elif max(abs(x))<4e2:
            self.scale_label=self.scales[5]
            self.scale=1e0
        elif max(abs(x))<4e5:
            self.scale_label=self.scales[6]
            self.scale=1e3
        elif max(abs(x))<4e8:
            self.scale_label=self.scales[7]
            self.scale=1e6
        elif max(abs(x))<4e11:
            self.scale_label=self.scales[8]
            self.scale=1e9
        else:
            self.scale_label=self.scales[9]
            self.scale=1e12
        return
    
    def set_varscalex(self,x):
        self.set_varscale(x)
        self.xscale_label=self.scale_label
        self.xscale=self.scale
        return
    
    def set_varscaley(self,y):
        self.set_varscale(y)
        self.yscale_label=self.scale_label
        self.yscale=self.scale
        return
    
    def set_varscalez(self,z):
        self.set_varscale(z)
        self.zscale_label=self.scale_label
        self.zscale=self.scale
        return
    
    def varscale(self,x):
        xscaled=x/self.scale
        return xscaled
    
    def varscalex(self,x):
        xscaled=x/self.xscale
        return xscaled
    
    def varscaley(self,y):
        yscaled=y/self.yscale
        return yscaled
    
    def varscalez(self,z):
        zscaled=z/self.zscale
        return zscaled
    
    def set_inset_varscalex(self,x):
        self.set_varscale(x)
        self.inset_xscale_label=self.scale_label
        self.inset_xscale=self.scale
        return
    
    def set_inset_varscaley(self,y):
        self.set_varscale(y)
        self.inset_yscale_label=self.scale_label
        self.inset_yscale=self.scale
        return
    
    def set_inset_varscalez(self,z):
        self.set_varscale(z)
        self.inset_zscale_label=self.scale_label
        self.inset_zscale=self.scale
        return
    
    def inset_varscalex(self,x):
        xscaled=x/self.inset_xscale
        return xscaled
    
    def inset_varscaley(self,y):
        yscaled=y/self.inset_yscale
        return yscaled
    
    def inset_varscalez(self,z):
        zscaled=z/self.inset_zscale
        return zscaled
    
    def pltappend(self,x,y,marker='.',mfc='k',mec='r',mew=1,linestyle='-',color=None,markersize=1,linewidth=1):
        
        xplot=self.varscalex(x)
        yplot=self.varscaley(y)
        
        self.ax.plot(xplot,yplot,marker=marker,mfc=mfc,mec=mec,mew=mew,linestyle=linestyle,c=color,markersize=markersize,linewidth=linewidth)

        return
    
    
    def convert3D_to_matrix(self,x,y,z):
        unique_values, indices = np.unique(x, return_index=True)
            
        if len(indices) > 1: # if first column has more than one unique value
            try:
                # shape of data
                l0 = np.sort(indices)[1]
                l1 = len(indices)
    
                # ignore last column if unfinished
                #print(l0*l1,len(x))
                if (l0*l1)!=x.size:
                    l1 = l1-1
                    #print(x)
            except:
                print("An error occurred") 
                    
                    
        X=np.reshape(x[:l0*l1], (l1,l0))
        Y=np.reshape(y[:l0*l1], (l1,l0))
        Z=np.reshape(z[:l0*l1], (l1,l0))
                    
                    
        return X,Y,Z
    
    def legend(self,legendArr,fontsize=6):
        self.ax.legend(legendArr,fontsize=fontsize)
    
    def addtext(self,text,x=0.02,y=0.88,fontsize=8.5,weight='bold',c='k'):
        fontsize=self.fontsizeout
        labelsize=self.labelsize
        self.fig.text(x,y,text,fontsize=fontsize,weight=weight,c=c)
    
    def addaxtext(self,text,x=0,y=0,fontsize=6,weight='normal',c='k'):
        fontsize=self.fontsizein
        labelsize=self.labelsize
        
        self.ax.text(x,y,text,fontsize=fontsize,weight=weight,c=c)
    
    def addaxtextmulti(self,i,text,x=0,y=0,fontsize=6,weight='normal',c='k'):
        fontsize=self.fontsizein
        labelsize=self.labelsize
        self.axs[i].text(x,y,text,fontsize=fontsize,weight=weight,c=c)
    
    def addinset(self,width="40%",height="40%",loc=1):
        self.axinset = inset_axes(self.ax, width=width, height=height, loc=loc)
        
    def plot2Dinset(self,x,y,xlabel,ylabel,xunit,yunit,marker='',mfc='k',mec='k',mew=0,linestyle='-',color=None,markersize=5,linewidth=1,fontsize=8.5,labelsize=6,label=None,rescale=True):
        fontsize=self.fontsize
        labelsize=self.labelsize
        
        # If first plot
        if self.inset_scaled==0 and rescale:
            self.set_inset_varscalex(x)
            self.set_inset_varscaley(y)
    
            self.axinset.set_xlabel(xlabel=xlabel+' ('+self.inset_xscale_label+xunit+')',fontsize=fontsize)
            self.axinset.set_ylabel(ylabel=ylabel+' ('+self.inset_yscale_label+yunit+')',fontsize=fontsize)
            self.axinset.tick_params(labelsize=labelsize,axis='x')
            self.axinset.tick_params(labelsize=labelsize,axis='y')
            self.inset_scaled=1 #Lock the scaling
        
        if rescale:
            xplot=self.inset_varscalex(x)
            yplot=self.inset_varscaley(y)
        else:
            xplot=x
            yplot=y
            self.axinset.set_xlabel(xlabel=xlabel+' ('+xunit+')',fontsize=fontsize)
            self.axinset.set_ylabel(ylabel=ylabel+' ('+yunit+')',fontsize=fontsize)
            self.axinset.tick_params(labelsize=labelsize,axis='x')
            self.axinset.tick_params(labelsize=labelsize,axis='y')
        
        if (mfc==None and mec==None):
            self.axinset.plot(xplot,yplot,marker=marker,mew=mew,linestyle=linestyle,markersize=markersize,linewidth=linewidth,c=color, label=label)
        else:
            self.axinset.plot(xplot,yplot,marker=marker,mfc=mfc,mec=mec,mew=mew,linestyle=linestyle, markersize=markersize,linewidth=linewidth,c=color, label=label)
        
        #plt.ticklabel_format(axis="x", style="plain", scilimits=(0,0))
        #plt.ticklabel_format(axis="y", style="plain", scilimits=(0,0))
        self.axinset.locator_params(axis='x', nbins=5)
        
        self.fig.tight_layout()
    
    
    def smooth(self,x,window_len=11,window='hanning'):
        """smooth the data using a window with requested size.
        
        This method is based on the convolution of a scaled window with the signal.
        The signal is prepared by introducing reflected copies of the signal 
        (with the window size) in both ends so that transient parts are minimized
        in the begining and end part of the output signal.
        
        input:
            x: the input signal 
            window_len: the dimension of the smoothing window; should be an odd integer
            window: the type of window from 'flat', 'hanning', 'hamming', 'bartlett', 'blackman'
                flat window will produce a moving average smoothing.
    
        output:
            the smoothed signal
            
        example:
    
        t=linspace(-2,2,0.1)
        x=sin(t)+randn(len(t))*0.1
        y=smooth(x)
        
        see also: 
        
        numpy.hanning, numpy.hamming, numpy.bartlett, numpy.blackman, numpy.convolve
        scipy.signal.lfilter
     
        TODO: the window parameter could be the window itself if an array instead of a string
        NOTE: length(output) != length(input), to correct this: return y[(window_len/2-1):-(window_len/2)] instead of just y.
        """
    
        if x.ndim != 1:
            raise ValueError("smooth only accepts 1 dimension arrays.")
    
        if x.size < window_len:
            raise ValueError("Input vector needs to be bigger than window size.")
    
    
        if window_len<3:
            return x
    
    
        if not window in ['flat', 'hanning', 'hamming', 'bartlett', 'blackman']:
            raise ValueError("Window is on of 'flat', 'hanning', 'hamming', 'bartlett', 'blackman'")
    
    
        s=np.r_[x[window_len-1:0:-1],x,x[-2:-window_len-1:-1]]
        #print(len(s))
        if window == 'flat': #moving average
            w=np.ones(window_len,'d')
        else:
            w=eval('np.'+window+'(window_len)')
    
        y=np.convolve(w/w.sum(),s,mode='valid')
        return y
